/**
* \file: string_set.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>

#include "utils/string_set.h"

typedef struct container_t container_t;


typedef struct container_t
{
	//the actual size of this container (sizeof(container_t)+sizeof(data))
	size_t container_size;

	//the data
	char data[];
} container_t;

typedef struct string_set_t
{
	size_t chunk_size;

	size_t buffer_size;

	size_t used_buffer_size;

	container_t first_container[];
} string_set_t;

static error_code_t string_set_reallocate(string_set_t **set, size_t buffer_size_needed);

//---------------------------------------------- API members --------------------------------------------------
error_code_t string_set_create(string_set_t **set_ptr, size_t chunk_size, unsigned int initial_chunk_cnt)
{
	size_t initial_buffer_size;
	string_set_t *set;

	//allocate buffer
	initial_buffer_size=sizeof(string_set_t)+chunk_size*initial_chunk_cnt;

	set=malloc(initial_buffer_size);
	if (set!=NULL)
	{
		set->buffer_size=initial_buffer_size;
		set->chunk_size=chunk_size;
		set->used_buffer_size=sizeof(string_set_t);
		(*set_ptr)=set;
	}
	else
		return RESULT_NORESOURCE;

	return RESULT_OK;
}

void string_set_free(string_set_t *set)
{
	free(set);
}

error_code_t string_set_add(string_set_t **set_ptr, const char *item)
{
	error_code_t result=RESULT_OK;

	size_t new_container_size;

	size_t buffer_size_needed;
	string_set_t *set;
	container_t *new_container;
	void *new_container_ptr;

	set=(*set_ptr);
	if (string_set_contains_item(set,item))
		return RESULT_OK;


	new_container_size=sizeof(container_t)+strlen(item)+1;


	//required buffer size: memory used by current containers + size for new container structure + size of data in the container
	buffer_size_needed=set->used_buffer_size+new_container_size;

	if (buffer_size_needed > set->buffer_size)
	{
		result=string_set_reallocate(set_ptr, buffer_size_needed);
		set=(*set_ptr);
	}

	if (result==RESULT_OK)
	{
		//put the element to the end of the used data block (ensured before that enough memory is present)
		//need this such complicated to satisfy lin(t)
		new_container_ptr=((char *)set)+set->used_buffer_size;
		new_container=(container_t *)new_container_ptr;


		//update the used buffer size taking care for the new element
		set->used_buffer_size=buffer_size_needed;
		//fill the new element with content
		new_container->container_size=new_container_size;
		strcpy(new_container->data, item);
	}

	return result;
}

bool string_set_contains_item(string_set_t *set, const char *item)
{
	container_t *container;
	container=string_set_first_item_container(set);
	while (container !=NULL)
	{
		if (strcmp(container->data, item)==0)
			return true;

		container=string_set_next_item_container(set, container);
	}

	return false;
}

bool string_set_does_item_match(string_set_t *set, const char *item)
{
	container_t *container;
	container=string_set_first_item_container(set);
	while (container !=NULL)
	{
		if (fnmatch(container->data, item,0)==0)
			return true;

		container=string_set_next_item_container(set, container);
	}

	return false;
}

container_t *string_set_first_item_container(string_set_t *set)
{
	//if our used buffersize is lower then needed to hold at least one element, we are empty
	if (set->used_buffer_size<(sizeof(string_set_t)+sizeof(container_t)))
		return NULL;
	return set->first_container;
}

container_t *string_set_next_item_container(string_set_t *set, container_t *cur_container)
{
	size_t used_size_calc;
	used_size_calc=(size_t)(((char *)cur_container)+cur_container->container_size-(char *)set);
	if (used_size_calc+sizeof(container_t)>set->used_buffer_size)
		return NULL;
	else
	{
		//need this such complicated to satisfy lin(t)
		void *new_container_ptr;
		new_container_ptr=(char *)cur_container+cur_container->container_size;
		return (container_t *)new_container_ptr;
	}
}

const char *string_set_get_item(container_t *container)
{
	return container->data;
}
//-------------------------------------------------------------------------------------------------------------

//---------------------------------------------- internal members ---------------------------------------------
static error_code_t string_set_reallocate(string_set_t **set_ptr, size_t buffer_size_needed)
{
	size_t new_size;
	string_set_t *new_set_ptr;
	string_set_t *cur_set_ptr;

	cur_set_ptr=(*set_ptr);

	new_size=((buffer_size_needed / cur_set_ptr->chunk_size)*cur_set_ptr->chunk_size);
	if (new_size<buffer_size_needed)
		new_size+=cur_set_ptr->chunk_size;
	new_set_ptr=realloc(cur_set_ptr, new_size);
	if (new_set_ptr!=NULL)
	{
		(*set_ptr)=new_set_ptr;
		(*set_ptr)->buffer_size=new_size;
		return RESULT_OK;
	}
	else
		return RESULT_NORESOURCE;
}
//-------------------------------------------------------------------------------------------------------------
